/*******************************************************************************
Copyright (c) 2010 Tecella LLC
Licensed under the MIT License
*******************************************************************************/

#include "system_monitor.h"
#include "ui_system_monitor.h"

#include <QMessageBox>

SystemMonitor::SystemMonitor(TECELLA_HNDL h, QWidget *parent) :
		QDialog(parent),
		ui(new Ui::SystemMonitor),
		h(h)
{
	ui->setupUi(this);

	connect( ui->pushButton_Exit, SIGNAL(clicked()), this, SLOT(close()) );
	connect( ui->spinBox_RefreshInterval, SIGNAL(valueChanged(int)), this, SLOT(updateTimerInterval(int)) );
	connect( &timer, SIGNAL(timeout()), this, SLOT(update())) ;

	updateTimerInterval( 200 );
}

SystemMonitor::~SystemMonitor()
{
	delete ui;
}

void SystemMonitor::changeEvent(QEvent *e)
{
	QDialog::changeEvent(e);
	switch (e->type()) {
	case QEvent::LanguageChange:
		ui->retranslateUi(this);
		break;
	default:
		break;
	}
}

void SystemMonitor::updateTimerInterval(int ms)
{
	if(ms==0) {
		timer.stop();
	} else {
		timer.start(ms);
	}
	ui->spinBox_RefreshInterval->setValue(ms);
}

void SystemMonitor::update()
{
	TECELLA_ERRNUM err;
	
	int result_count;
	int results_per_controller;
	int results_per_slot;

	//Run the system monitor once
	err = tecella_system_monitor_run(
		h, &result_count, &results_per_controller, &results_per_slot );
	if(err!=TECELLA_ERR_OK)
	{
		timer.stop();
		return;
	}

	//Diable parts that aren't applicable
	ui->groupBox_MainBoard->setEnabled(results_per_controller!=0);
	ui->groupBox_DaughterBoards->setEnabled(results_per_slot!=0);
	ui->label_DaughterBoardsNA->setShown(results_per_slot==0 );
	if( results_per_controller==0 && results_per_slot==0 ) {
		QMessageBox::information(this, "System Monitor", "System Monitor is not currently supported with your amplifier.");
		close();
	}

	//Get some amplifier properties
	TECELLA_HW_PROPS hwprops;
	err = tecella_get_hw_props(h, &hwprops);
	if(err!=TECELLA_ERR_OK)
	{
		timer.stop();
		return;
	}

	QTableWidgetItem *item;

	//Initialize the controller board table if necessary
	QTableWidget *table_m = ui->tableWidget_MainBoard;
	if( table_m->columnCount() < 1 || table_m->rowCount() < results_per_controller )
	{
		//Setup the column for the controller board
		table_m->setColumnCount(1);
		if( hwprops.nslots!=0 ) {
			item = new QTableWidgetItem("Slot 0");
		} else {
			item = new QTableWidgetItem("");
		}
		item->setFont( QFont("Courier New",10) );
		table_m->setHorizontalHeaderItem(0, item );

		//Setup the rows for the controller board
		table_m->setRowCount(results_per_controller);
		for(int i=0; i<results_per_controller; ++i)
		{
			//row header
			item = new QTableWidgetItem();
			item->setFont( QFont("Courier New",8) );
			item->setTextAlignment( Qt::AlignRight | Qt::AlignVCenter );
			table_m->setVerticalHeaderItem(i, item );

			//data
			item = new QTableWidgetItem();
			item->setFont( QFont("Courier New",8) );
			item->setTextAlignment( Qt::AlignRight | Qt::AlignVCenter );
			table_m->setItem(i, 0, item);
		}
	}

	//Initialize the channel board table if necessary.
	QTableWidget *table_d = ui->tableWidget_DaughterBoards;
	if( table_d->columnCount() < hwprops.nslots || table_d->rowCount() < results_per_slot )
	{
		//Setup the column for the controller board
		table_d->setColumnCount(hwprops.nslots);
		for(int i=0; i<hwprops.nslots; ++i)
		{
			item = new QTableWidgetItem( "Slot " + QString::number(i+1) );
			item->setFont( QFont("Courier New",8) );
			item->setTextAlignment( Qt::AlignRight | Qt::AlignVCenter );
			table_d->setHorizontalHeaderItem(i, item );
		}

		//Setup the rows for the controller board
		table_d->setRowCount(results_per_slot);
		for(int i=0; i<results_per_slot; ++i)
		{
			item = new QTableWidgetItem();
			item->setFont( QFont("Courier New",8) );
			item->setTextAlignment( Qt::AlignRight | Qt::AlignVCenter );
			table_d->setVerticalHeaderItem(i, item );
		}

		//Setup the body of the table
		for(int i=0; i<hwprops.nslots; ++i)
		{
			for(int j=0; j<results_per_slot; ++j)
			{
				item = new QTableWidgetItem();
				item->setFont( QFont("Courier New",8) );
				item->setTextAlignment( Qt::AlignRight );
				table_d->setItem(j, i, item);
			}
		}
	}

	//Iterate through each result and update the tables
	for(int r=0; r<result_count; ++r)
	{
		//Get the result
		TECELLA_SYSTEM_MONITOR_RESULT smr;
		err = tecella_system_monitor_get(h, r, &smr);
		if(err!=TECELLA_ERR_OK)
		{
			timer.stop();
			return;
		}

		//Main Board
		if( smr.board_index == 0 )
		{
			int row = smr.board_result_index;
			ui->tableWidget_MainBoard->verticalHeaderItem(row)->setText(
					QString::fromWCharArray(smr.label) );
			ui->tableWidget_MainBoard->item(row, 0)->setText(
					QString::number(smr.value,'f',3) + QString::fromWCharArray(smr.units) );
		}

		//Daughter Boards
		else
		{
			int col = smr.board_index-1;
			int row = smr.board_result_index;
			ui->tableWidget_DaughterBoards->verticalHeaderItem(row)->setText(
					QString::fromWCharArray(smr.label) );
			ui->tableWidget_DaughterBoards->item(row, col)->setText(
					QString::number(smr.value,'f',3) + QString::fromWCharArray(smr.units) );
		}
	}

	ui->tableWidget_MainBoard->adjustCellSizes();
	ui->tableWidget_DaughterBoards->adjustCellSizes();
}
