/******************************************************************************
Tecella Lossless Compression Format
Copyright (c) 2010 Tecella LLC
Licensed under the MIT License

*******************************************************************************
File Description:
Provides the C interface for reading and writing TLC files.
Pretty much a wrapper around the C++ classes tlc_writer and tlc_reader.

*******************************************************************************
Contributors:
Brian Anderson (Tecella) - Initial implementation

******************************************************************************/

#define DLLEXPORT __declspec(dllexport)

#include "TecellaTLC.h"
#include "tlc.h"

#include <assert.h>
#include <map>
using namespace std;

#define TLC_VERSION_CURRENT 1

map <TLC_WRITE_HANDLE, tlc_writer*> tlc_writers;
TLC_WRITE_HANDLE new_write_handle = 1;
map <TLC_READ_HANDLE, tlc_reader*> tlc_readers;
TLC_READ_HANDLE new_read_handle = 1;

bool tlcw_handle_valid( TLC_WRITE_HANDLE h )
{
	if( tlc_writers.count(h) == 1 ) {
		return true;
	} else {
		assert(false);
		return false;
	}
}

extern "C" DLLEXPORT
int CALL tlcw_initialize( TLC_WRITE_HANDLE *h, const char *filename,
                         unsigned short channel_count,
                         bool interpret_as_period, double period_or_frequency,
                         unsigned char bits_per_sample, unsigned int samples_per_frame )
{
	return tlcw_initialize_old(h, TLC_VERSION_CURRENT, filename, channel_count, interpret_as_period, period_or_frequency, bits_per_sample, samples_per_frame);
}

extern "C" DLLEXPORT
int CALL tlcw_initialize_old( TLC_WRITE_HANDLE *h, int version, const char *filename,
                         unsigned short channel_count,
                         bool interpret_as_period, double period_or_frequency,
                         unsigned char bits_per_sample, unsigned int samples_per_frame )
{
	while( tlc_writers.count(new_write_handle) != 0 ) {
		++new_write_handle;
	}

	*h = new_write_handle;
	tlc_writers[new_write_handle] =
		new tlc_writer( version, filename, channel_count,
		                interpret_as_period, period_or_frequency,
				bits_per_sample, samples_per_frame);
	return 0;
}

extern "C" DLLEXPORT
int CALL tlcw_set_channel_scale(TLC_WRITE_HANDLE h, int channel, double scale)
{
	if( !tlcw_handle_valid(h) ) return 1;  //check handle is valid

	tlc_writer *tlcw = tlc_writers[h];
	tlcw->set_channel_scale(channel, scale);
	return 0;
}

extern "C" DLLEXPORT
int CALL tlcw_set_channel_label(TLC_WRITE_HANDLE h, int channel, const char *label, unsigned int length)
{
	if( !tlcw_handle_valid(h) ) return 1;  //check handle is valid

	tlc_writer *tlcw = tlc_writers[h];
	tlcw->set_channel_label(channel, label, length);
	return 0;
}

extern "C" DLLEXPORT
int CALL tlcw_set_metadata(TLC_WRITE_HANDLE h, char *data, unsigned int length)
{
	if( !tlcw_handle_valid(h) ) return 1;  //check handle is valid

	tlc_writer *tlcw = tlc_writers[h];
	tlcw->set_metadata(data, length);
	return 0;
}

extern "C" DLLEXPORT
int CALL tlcw_start_writing_frames(TLC_WRITE_HANDLE h)
{
	if( !tlcw_handle_valid(h) ) return 1;  //check handle is valid

	tlc_writer *tlcw = tlc_writers[h];
	tlcw->start_writing_frames();
	return 0;
}

extern "C" DLLEXPORT
int CALL tlcw_write_samples(TLC_WRITE_HANDLE h, int channel, short *data, unsigned int length)
{
	if( !tlcw_handle_valid(h) ) return 1;  //check handle is valid

	tlc_writer *tlcw = tlc_writers[h];
	tlcw->write_samples(channel, data, length);
	return 0;
}

extern "C" DLLEXPORT
int CALL tlcw_finalize(TLC_WRITE_HANDLE h)
{
	if( !tlcw_handle_valid(h) ) return 1;  //check handle is valid

	tlc_writer *tlcw = tlc_writers[h];
	tlcw->finalize();
	tlc_writers.erase(h);
	delete tlcw;
	return 0;
}



extern "C" DLLEXPORT
int CALL tlcr_open( TLC_READ_HANDLE *h, const char *filename )
{
	while( tlc_readers.count(new_read_handle) != 0 ) {
		++new_read_handle;
	}

	*h = new_read_handle;
	tlc_readers[new_read_handle] = new tlc_reader(filename);
	return 0;	
}

extern "C" DLLEXPORT
int CALL tlcr_close( TLC_READ_HANDLE h )
{
	tlc_reader *tlcr = tlc_readers[h];
	tlc_readers.erase(h);
	delete tlcr;
	return 0;
}

extern "C" DLLEXPORT
int CALL tlcr_get_info( TLC_READ_HANDLE h, TLC_INFO *info )
{
	tlc_reader *tlcr = tlc_readers[h];
	const tlc_file_header &tlch = tlcr->getHeader();

	info->tlc_version = tlch.tlc_version;
	info->channel_count = tlch.channel_count;
	
	if(tlch.freq_or_period_mode == 0) {
		info->interpret_as_period = true;
		info->interpret_as_frequency = false;
		info->period = tlch.freq_or_period;
		info->frequency = 1.0 / info->period;
	}
	else {
		info->interpret_as_period = false;
		info->interpret_as_frequency = true;
		info->frequency = tlch.freq_or_period;
		info->period = 1.0 / info->frequency;
	}

	info->bits_per_sample = tlch.bits_per_sample;
	info->samples_per_channel = tlch.samples_per_channel;
	info->samples_per_frame = tlch.samples_per_frame;
	info->frame_index_offset = tlch.frame_index_offset;
	//info->metadata_size = (unsigned int) tlch.metadata.size();
	return 0;
}

extern "C" DLLEXPORT
int CALL tlcr_get_metadata( TLC_READ_HANDLE h, char *data, unsigned int max_length )
{
	tlc_reader *tlcr = tlc_readers[h];
	const tlc_file_header &tlch = tlcr->getHeader();
	memcpy(data, &tlch.metadata[0], min(max_length, tlch.metadata.size()) );
	return 0;
}

extern "C" DLLEXPORT
int CALL tlcr_get_channel_scale( TLC_READ_HANDLE h, int channel, double *scale )
{
	tlc_reader *tlcr = tlc_readers[h];
	const tlc_file_header &tlch = tlcr->getHeader();
	*scale = tlch.channel_scales[channel];
	return 0;
}

extern "C" DLLEXPORT
int CALL tlcr_get_channel_label( TLC_READ_HANDLE h, int channel, char *label, unsigned int max_length )
{
	tlc_reader *tlcr = tlc_readers[h];
	const tlc_file_header &tlch = tlcr->getHeader();
	strncpy(label, tlch.channel_labels[channel].c_str(), tlch.channel_labels[channel].length() );
	return 0;
}

extern "C" DLLEXPORT
int CALL tlcr_read_i( TLC_READ_HANDLE h,
					 int channel,
					 unsigned long long first_sample_index,
					 unsigned int length,
					 short *data )
{
	tlc_reader *tlcr = tlc_readers[h];
	tlcr->read(channel, first_sample_index, length, data);
	return 0;
}

extern "C" DLLEXPORT
int CALL tlcr_read_d( TLC_READ_HANDLE h,
					 int channel,
					 unsigned long long first_sample_index,
					 unsigned int length,
					 double *data )
{
	tlc_reader *tlcr = tlc_readers[h];
	vector<short> data_i(length);
	tlcr->read(channel, first_sample_index, length, &data_i[0]);

	double scale = tlcr->getHeader().channel_scales[channel];
	for(unsigned int i=0; i<length; i++) {
		data[i] = data_i[i] * scale;
	}

	return 0;
}

//extern "C" DLLEXPORT

