/******************************************************************************
Tecella Lossless Compression Format
Copyright (c) 2010 Tecella LLC
Licensed under the MIT License

*******************************************************************************
File Description:
Provides the C interface for reading and writing TLC files.

*******************************************************************************
Contributors:
Brian Anderson (Tecella) - Initial implementation

******************************************************************************/


#ifndef __TECELLA_TLC__
#define __TECELLA_TLC__

#ifndef DLLEXPORT
#define DLLEXPORT
#endif

#define CALL __cdecl

#ifdef __cplusplus
extern "C" {
#endif

/******************************************************************************
* Writing/Encoding functions
******************************************************************************/

/** The TLC writer handle type */
typedef unsigned int TLC_WRITE_HANDLE;

/** Initializes a TLC file with the requested properties.  Always uses the newest TLC file format.
@param h A handle to the new encoder is returned in h.
@param filename The filename of the new TLC file.  If filename exists, it will be overwritten.
@param channel_count The number of channels in the file.
@param interpret_as_period Indicates whether period_or_frequency should be interpreted as period.  False implies interperet as frequency.
@param period_or_frequency Sample period or frequency.  Units for period is seconds and units for frequency is Hz.
@param bits_per_sample Number of bits per sample.  Max supported by this API is 16.  If fewer bits are used, file size will be reduced.
@param samples_per_frame  A file can be accessed randomly to the first of every N samples, where N = samples_per_frame.  An entire frame must be decoded to access the last of every N samples.  Smaller frame sizes make random access faster, but larger frame sizes reduce overall file size.  A good rule of thumb is to set samples_per_frame to the number of samples in a second for a single channel.
*/
DLLEXPORT int CALL tlcw_initialize( TLC_WRITE_HANDLE *h, const char *filename,
                                    unsigned short channel_count,
                                    bool interpret_as_period, double period_or_frequency,
                                    unsigned char bits_per_sample, unsigned int samples_per_frame );

/** Initializes a TLC file with the requested properties.  Same as tlcw_initialize(), but allows you to specify the exact TLC version you wish to use.
@param h A handle to the new encoder is returned in h.
@param version The version of TLC file to create.  v0 has 5% better compression than v1, but is slower, more complicated, and not parallelizable in a fine-grain manner.  v1 has about 5% worse compression than v0, but is faster, simpler, and parallelizable in a coarse and fine-grain manner.
@param filename The filename of the new TLC file.  If filename exists, it will be overwritten.
@param channel_count The number of channels in the file.
@param interpret_as_period Indicates whether period_or_frequency should be interpreted as period.  False implies interperet as frequency.
@param period_or_frequency Sample period or frequency.  Units for period is seconds and units for frequency is Hz.
@param bits_per_sample Number of bits per sample.  Max supported by this API is 16.  If fewer bits are used, file size will be reduced.
@param samples_per_frame  A file can be accessed randomly to the first of every N samples, where N = samples_per_frame.  An entire frame must be decoded to access the last of every N samples.  Smaller frame sizes make random access faster, but larger frame sizes reduce overall file size.  A good rule of thumb is to set samples_per_frame to the number of samples in a second for a single channel.
*/
DLLEXPORT int CALL tlcw_initialize_old( TLC_WRITE_HANDLE *h, int version, const char *filename,
                                        unsigned short channel_count,
                                        bool interpret_as_period, double period_or_frequency,
                                        unsigned char bits_per_sample, unsigned int samples_per_frame );

/** Sets the scale for a given channel.
Must be called before tlcw_start_writing_frames().
@param h A handle to a previously initialized encoder.
@param channel The channel for which to set the scale.
@param scale The floating point value to multiply the raw integral sample by to get the proper units (which can be specified in the label).
*/
DLLEXPORT int CALL tlcw_set_channel_scale(TLC_WRITE_HANDLE h, int channel, double scale);

/** Sets the label for a given channel.
Must be called before tlcw_start_writing_frames().
@param h A handle to a previously initialized encoder.
@param channel The channel for which to set the label.
@param label The channel's label.  Do not null terminate.
@param length The number of characters in label.  Do not include any null termination.
*/
DLLEXPORT int CALL tlcw_set_channel_label(TLC_WRITE_HANDLE h, int channel, const char *label, unsigned int length);

/** Indicates that file initialization is complete and samples will be written.
Calls to tlcw_set_channel_scale() and tlcw_set_channel_label() cannot occur after calling this function.
@param h A handle to a previously initialized encoder.
*/
DLLEXPORT int CALL tlcw_start_writing_frames(TLC_WRITE_HANDLE h);

/** Writes a block of samples to the given channel.
Can only be called after tlcw_start_writing_frames() has been called.
Note: The encoder cannot encode a frame until it has samples_per_frame samples from all the channels.  The API currently keeps an internal copy of all data until it can be encoded.  Data should be written to all channels at the same rate or you will eventually run out of memory.
@param h A handle to a previously initialized encoder.
@param channel The channel for which to write the data.
@param data An array of samples.
@param length The number of samples in data.
*/
DLLEXPORT int CALL tlcw_write_samples(TLC_WRITE_HANDLE h, int channel, short *data, unsigned int length);

/** Writes the frame index for random read access and closes the file.
Note: If a any channel has more data written to it than others, that data will be discarded.
@param h A handle to a previously initialized encoder.
*/
DLLEXPORT int CALL tlcw_finalize(TLC_WRITE_HANDLE h);

/******************************************************************************
* Reading/Decoding functions
******************************************************************************/

/** The TLC reader handle type */
typedef unsigned int TLC_READ_HANDLE;

typedef struct tlc_info
{
	unsigned char  tlc_version;                /**< The version of the TLC file format used by the encoder. */
	unsigned short channel_count;              /**< The number of channels in the file. */
	bool           interpret_as_period;        /**< Indicates if sample duration was encoded as a period in the file. */
	bool           interpret_as_frequency;     /**< Indicates if sample duration was encoded as a frequency in the file. */
	double         period;                     /**< Seconds. If interpret_as_period, the period in the file is copied directly.  Otherwise period = 1/frequency. */
	double         frequency;                  /**< Hz. If interpret_as_frequency, the frequency in the file is copied directly.  Otherwise frequency = 1/period. */
	unsigned char  bits_per_sample;            /**< The number of bits per sample.  Max supported by this API is currently 16bps. */
	unsigned long long  samples_per_channel;   /**< The number of samples availabe for each channel. */
	unsigned int   samples_per_frame;          /**< The number of samples per frame of the given file. See tlcw_initialize() for a description. */
	unsigned long long  frame_index_offset;    /**< The offset of the frame index used for random access.  Zero, indicates file was not closed properly and needs to be repaired.  A repair function is not yet available in this version. */
} TLC_INFO;

/** Opens a TLC file for decode.
@param h A handle to the new decoder is returned in h.
@param filename A null terminated string of the filename to open.
*/
DLLEXPORT int CALL tlcr_open( TLC_READ_HANDLE *h, const char *filename );

/** Closes a TLC file
@param h A handle to a previously opened decoder.
*/
DLLEXPORT int CALL tlcr_close( TLC_READ_HANDLE h );

/** Returns info about an opened TLC file.
@param h A handle to a previously opened decoder.
@param info The address of the structure in which the file info will be returned.
*/
DLLEXPORT int CALL tlcr_get_info( TLC_READ_HANDLE h, TLC_INFO *info );

/** Copies the files metadata to memory
@param h A handle to a previously opened decoder.
@param data A pointer to which the data will be copied.
@param max_length The max size of data that can be copied (in bytes).  (To prevent buffer overflows.)
*/
DLLEXPORT int CALL tlcr_get_metadata( TLC_READ_HANDLE h, char *data, unsigned int max_length );

/** Retrieves the scale for a given channel.
Each sample returned by tlcr_read_i() should be multiplied by the scale to get the proper units.
@param h A handle to a previously opened decoder.
@param channel The channel to get the scale of.
@param scale The scale will be returned using this argument.
*/
DLLEXPORT int CALL tlcr_get_channel_scale( TLC_READ_HANDLE h, int channel, double *scale );

/** Retrieves the label for a given channel.
Each sample returned by tlcr_read_i() should be multiplied by the scale to get the proper units.
@param h A handle to a previously opened decoder.
@param channel The channel to get the label of.
@param label The channel's label will be copied here.
@param max_length The maximum number of chars that can be copied to label. 
*/
DLLEXPORT int CALL tlcr_get_channel_label( TLC_READ_HANDLE h, int channel, char *label, unsigned int max_length );

/** Retrieves a set of raw integral samples for a given channel.
Use the scale returned by tlcr_get_channel_scale() to convert each sample to the proper units.
@param h A handle to a previously opened decoder.
@param first_sample_index The first sample to decode.
@param length How many samples to decode.
@param data The data will be decoded into this buffer.
*/
DLLEXPORT int CALL tlcr_read_i( TLC_READ_HANDLE h, int channel, unsigned long long first_sample_index, unsigned int length, short *data );

/** Retrieves a set of properly scaled samples for a given channel.
tlcr_get_channel_scale() is not needed if using this function, but the samples will take 4x the memory compared to tlcr_read_i.
@param h A handle to a previously opened decoder.
@param first_sample_index The first sample to decode.
@param length How many samples to decode.
@param data The data will be decoded into this buffer.
*/
DLLEXPORT int CALL tlcr_read_d( TLC_READ_HANDLE h, int channel, unsigned long long first_sample_index, unsigned int length, double *data );


#ifdef __cplusplus
}
#endif

#endif
