/******************************************************************************
Tecella Lossless Compression Format
Copyright (c) 2010 Tecella LLC
Licensed under the MIT License

*******************************************************************************
File Description:
Provides wrappers around the Windows API for large file (>2GB) access.
Implements the basic functionality of ofstream (via tc_oftream) and
   ifstream (via tc_ifstream).
These classes should be used in place of ofstream and ifstream on
   Windows because large file support by ofstream and ifstream in
   Visual Studio and MinGW is spotty.

*******************************************************************************
Contributors:
Brian Anderson (Tecella) - Initial implementation

******************************************************************************/

#ifndef __TC_FSTREAM__
#define __TC_FSTREAM__

#include <windows.h>

#include <iostream>
#include <fstream>
using namespace std;

#ifndef TCD
#define TCD
#endif

class tc_ofstream
{
private:
	HANDLE file_handle;

public:
	tc_ofstream()
	{
TCD		file_handle = INVALID_HANDLE_VALUE;
	}

	tc_ofstream( const char *filename,
	             ios_base::openmode mode=ios_base::out )
	{
TCD		file_handle = INVALID_HANDLE_VALUE;
		open(filename);
	}

	tc_ofstream( const tc_ofstream &src )
	{
TCD		file_handle = src.file_handle;
	}
	
	~tc_ofstream()
	{
		close();
	}

	bool is_open()
	{
TCD		return (file_handle!=INVALID_HANDLE_VALUE);
	}

	void open( const char *filename,
	           ios_base::openmode mode=ios_base::out)
	{
TCD		if(file_handle!=INVALID_HANDLE_VALUE) {
			close();
		}

		file_handle = CreateFile( filename,
			GENERIC_WRITE, FILE_SHARE_READ, NULL,
			CREATE_ALWAYS, FILE_ATTRIBUTE_NORMAL, NULL);

		if(file_handle == INVALID_HANDLE_VALUE)
		{
			//error
TCD			return;
		}
	}

	void close()
	{
TCD		if(file_handle!=INVALID_HANDLE_VALUE) {
			CloseHandle(file_handle);
			file_handle = INVALID_HANDLE_VALUE;
		}
	}

	tc_ofstream& write(const char *data, streamsize size)
	{
		DWORD ret_size;
		WriteFile( file_handle, data, size, &ret_size, NULL );
TCD		return *this;
	}

	bool flush()
	{
		return FlushFileBuffers(file_handle);
	}

	streampos tellp()
	{
TCD		LARGE_INTEGER OldFilePointer;
		OldFilePointer.QuadPart = 0;
		LARGE_INTEGER NewFilePointer;
		SetFilePointerEx( file_handle, OldFilePointer,
				&NewFilePointer, FILE_CURRENT );
		return NewFilePointer.QuadPart;
	}

	tc_ofstream& seekp(streampos pos)
	{
TCD		LARGE_INTEGER NewFilePointer;
		NewFilePointer.QuadPart = pos;
		SetFilePointerEx(file_handle, NewFilePointer, NULL, FILE_BEGIN);
		return *this;
	}

	tc_ofstream& seekp(streamoff off, ios_base::seekdir dir)
	{
TCD		DWORD MoveMethod = 0;
		switch(dir) {
		case ios_base::beg: MoveMethod = FILE_BEGIN; break; 
		case ios_base::cur: MoveMethod = FILE_CURRENT; break;
		case ios_base::end: MoveMethod = FILE_END; break;
		default: break;
		}

		LARGE_INTEGER NewFilePointer;
		NewFilePointer.QuadPart = off;
		SetFilePointerEx(file_handle, NewFilePointer, NULL, MoveMethod);
		return *this;
	}
};


class tc_ifstream
{
private:
	HANDLE file_handle;

public:
	tc_ifstream()
	{
TCD		file_handle = INVALID_HANDLE_VALUE;
	}

	tc_ifstream( const char *filename,
	             ios_base::openmode mode=ios_base::in )
	{
TCD		file_handle = INVALID_HANDLE_VALUE;
		open(filename);
	}

	tc_ifstream( const tc_ifstream &src )
	{
TCD		file_handle = src.file_handle;
	}

	~tc_ifstream()
	{
		close();
	}

	bool is_open()
	{
TCD		return (file_handle!=INVALID_HANDLE_VALUE);
	}

	void open( const char *filename,
	           ios_base::openmode mode=ios_base::out)
	{
TCD		if(file_handle==INVALID_HANDLE_VALUE) {
			close();
		}

		file_handle = CreateFile( filename,
			GENERIC_READ, FILE_SHARE_READ, NULL,
			OPEN_EXISTING, FILE_ATTRIBUTE_NORMAL, NULL);

		if(file_handle == INVALID_HANDLE_VALUE)
		{
			//error
			return;
		}
	}

	void close()
	{
TCD		if(file_handle!=INVALID_HANDLE_VALUE) {
			CloseHandle(file_handle);
			file_handle = INVALID_HANDLE_VALUE;
		}
	}

	tc_ifstream& read(char *data, streamsize size)
	{
		DWORD ret_size;
TCD		ReadFile( file_handle, data, size, &ret_size, NULL );
		return *this;
	}

	streampos tellg()
	{
TCD		LARGE_INTEGER OldFilePointer;
		OldFilePointer.QuadPart = 0;
		LARGE_INTEGER NewFilePointer;
		SetFilePointerEx(file_handle, OldFilePointer,
				&NewFilePointer, FILE_CURRENT);
		return NewFilePointer.QuadPart;
	}

	tc_ifstream& seekg(streampos pos)
	{
TCD		LARGE_INTEGER NewFilePointer;
		NewFilePointer.QuadPart = pos;
		if( !SetFilePointerEx(file_handle, NewFilePointer, NULL, FILE_BEGIN) )
		{
			
		}
		return *this;
	}

	tc_ifstream& seekg(streamoff off, ios_base::seekdir dir)
	{
TCD		DWORD MoveMethod = 0;
		switch(dir) {
		case ios_base::beg: MoveMethod = FILE_BEGIN; break; 
		case ios_base::cur: MoveMethod = FILE_CURRENT; break;
		case ios_base::end: MoveMethod = FILE_END; break;
		default: break;
		}

		LARGE_INTEGER NewFilePointer;
		NewFilePointer.QuadPart = off;
		SetFilePointerEx(file_handle, NewFilePointer, NULL, MoveMethod);
		return *this;
	}

	bool eof() {
TCD		LARGE_INTEGER FileSize;
		GetFileSizeEx(file_handle, &FileSize);
		return tellg() >= FileSize.QuadPart;
	}

	void clear(ios_base::iostate state = ios_base::goodbit) {
	}
};


#endif

