/******************************************************************************
Tecella Lossless Compression Format
Copyright (c) 2010 Tecella LLC
Licensed under the MIT License

*******************************************************************************
File Description:
Provides the framework for different versions of the TLC file format.
Also manages the different versions.

*******************************************************************************
Contributors:
Brian Anderson (Tecella) - Initial implementation

******************************************************************************
TODO:
1) Code assumes platform uses little endian byte order.  Don't do that.
******************************************************************************/

#include "tlc.h"
#include "tlc_v0.h"
#include "tlc_v1.h"

#include <assert.h>
using namespace std;

unsigned int *dbg_spf;

mutex cout_mutex;

//defining tss_cleanup_implemented should
// disable TLS (thread local storage) callbacks in boost::threads
//In turn this should enable UPX compression of TecellaTLC.dll.
extern "C" void tss_cleanup_implemented(void)
{
	return;
}

////////////////////////////////////
//Various utility functions

//returns the minimum number of bits needed to represent i in two's complement
unsigned char bits_needed(int i)
{
	if(i<0) i = -(i+1);
	unsigned char bits = 1;
	while(i)
	{
		i>>=1;
		bits++;
	}
	return bits;
}

unsigned char bits_needed_v1(int i)
{
	if(i==0) return 0;
	if(i<0) i = -(i+1);
	unsigned char bits = 1;
	while(i)
	{
		i>>=1;
		bits++;
	}
	return bits;
}


/******************************************************************************
* tlc_file_header
*/
bool tlc_file_header::write(ofstream &ofs)
{
	ofs.write("TLCave",6);
	ofs.write( (char*)&tlc_version, sizeof(tlc_version) );
	ofs.write( (char*)&channel_count, sizeof(channel_count) );
	ofs.write( (char*)&freq_or_period_mode, sizeof(freq_or_period_mode) );
	ofs.write( (char*)&freq_or_period, sizeof(freq_or_period) );
	ofs.write( (char*)&bits_per_sample, sizeof(bits_per_sample) );
	ofs.write( (char*)&samples_per_channel, sizeof(samples_per_channel) );
	ofs.write( (char*)&samples_per_frame, sizeof(samples_per_frame) );
	ofs.write( (char*)&frame_index_offset, sizeof(frame_index_offset) );

	//metadata
	unsigned int metadata_size = (unsigned int) metadata.size();
	ofs.write( (char*)&metadata_size, sizeof(metadata_size) );
	if(metadata_size>0) {
		ofs.write( (char*)&metadata[0], metadata_size );
	}
	
	//per channel header
	for(int c=0; c<channel_count; c++)
	{
		//write channel scale data
		ofs.write( (char*)&channel_scales[c], sizeof(channel_scales[c]) );

		//write label data
		unsigned int label_length = (unsigned int) channel_labels[c].length();
		unsigned char label_length_truncated = 255;
		if(label_length<255) label_length_truncated = (unsigned char) label_length;

		ofs.write( (char*)&label_length_truncated, sizeof(label_length_truncated) );
		ofs.write( (char*)channel_labels[c].c_str(), label_length_truncated );
	}

	return true;
}

bool tlc_file_header::read(ifstream &ifs)
{	
	ifs.seekg(0);
	char tlcave[6];
	ifs.read( (char*)&tlcave,6 );
	if( strncmp("TLCave",tlcave,6) != 0 )
		return false; //ERROR
	
	ifs.read( (char*)&tlc_version, sizeof(tlc_version) );
	if( tlc_version!=0 && tlc_version!=1 )
		return false; //ERROR

	ifs.read( (char*)&channel_count, sizeof(channel_count) );
	ifs.read( (char*)&freq_or_period_mode, sizeof(freq_or_period_mode) );
	ifs.read( (char*)&freq_or_period, sizeof(freq_or_period) );
	ifs.read( (char*)&bits_per_sample, sizeof(bits_per_sample) );
	ifs.read( (char*)&samples_per_channel, sizeof(samples_per_channel) );
	ifs.read( (char*)&samples_per_frame, sizeof(samples_per_frame) );
	ifs.read( (char*)&frame_index_offset, sizeof(frame_index_offset) );

	//metadata
	unsigned int metadata_size = 0;
	ifs.read( (char*)&metadata_size, sizeof(metadata_size) );
	metadata.resize(metadata_size);
	if(metadata_size>0) {
		ifs.read( (char*)&metadata[0], metadata_size );
	}

	channel_scales.resize(channel_count);
	channel_labels.resize(channel_count);
	for(int c=0; c<channel_count; c++)
	{
		//read channel scale data
		ifs.read( (char*)&channel_scales[c], sizeof(channel_scales[c]) );

		//read label data
		char label[255];
		unsigned char label_length;
		ifs.read( (char*)&label_length, sizeof(label_length) );
		channel_labels[c].resize(label_length);
		ifs.read( (char*)label, label_length );
		channel_labels[c].assign(label,label_length);
	}

	return true;
}

/******************************************************************************
* tlc_writer
*/
tlc_writer::tlc_writer( unsigned int version,
                        const char* filename,
                        unsigned short channel_count,
                        bool interpret_as_period,
                        double  sample_freq_or_period,
						unsigned char  bits_per_sample,
				        unsigned int   samples_per_frame )
#ifdef TLC_USE_FRAME_WRITE_THREAD_POOL
: frame_thread_pool( thread::hardware_concurrency() )
#endif
{
	dbg_spf = &header.samples_per_frame;

	init_error = false;
	if(version!=0 && version!=1) {
		init_error = true;
		return;
	}

	if(version==1)
	{
		//make sure samples_per_frame is a valid multiple
		//samples_per_frame = ((samples_per_frame/TLC_V1_BLOCK_SIZE)*TLC_V1_BLOCK_SIZE) + 1;
	}

	samples_per_channel_file_ptr = 6 + //size of magic string
	                      sizeof(header.tlc_version) +
	                      sizeof(header.channel_count) +
						  sizeof(header.freq_or_period_mode) +
	                      sizeof(header.freq_or_period) +
	                      sizeof(header.bits_per_sample);

	frame_index_offset_file_ptr = samples_per_channel_file_ptr +
	                      sizeof(header.samples_per_channel) +
	                      sizeof(header.samples_per_frame);

	header.tlc_version = version;
	header.channel_count = channel_count;
	header.freq_or_period_mode = interpret_as_period ? 0 : 1;
	header.freq_or_period = sample_freq_or_period;
	header.bits_per_sample = bits_per_sample;
	header.samples_per_channel = 0;
cout << "samples per frame = " << samples_per_frame << endl;
	header.samples_per_frame = samples_per_frame;
	header.frame_index_offset = 0;
	header.channel_scales.resize(channel_count, 1.0);
	header.channel_labels.resize(channel_count);
	for(int i=0; i<channel_count; i++)
	{
		char temp[16];
		sprintf(temp,"Ch%d",i+1);
		header.channel_labels[i] = temp;
	}

	ofs.open(filename, ios::binary | ios::out);

	frames_queue.resize(channel_count);
	finish_write_thread = false;
}

tlc_writer::~tlc_writer()
{
	finalize();
}

bool tlc_writer::set_metadata(char *data, unsigned int length)
{
	header.metadata.resize(length);
	memcpy(&header.metadata[0], data, length);
	return true;
}

bool tlc_writer::set_channel_scale(int channel, double scale)
{
	if(channel<0 || channel>=header.channel_count) return false;
	header.channel_scales[channel] = scale;
	return true;
}

bool tlc_writer::set_channel_label(int channel, const char *label, int length)
{
	if(channel<0 || channel>=header.channel_count) return false;
	header.channel_labels[channel].assign(label,length);
	return true;
}

bool  tlc_writer::start_writing_frames()
{
	header.write(ofs);
#ifdef TLC_USE_FILE_WRITE_THREAD
	file_write_thread = thread( std::ref(*this) );
#endif
	return true;
}

tlc_frame* tlc_writer::new_empty_frame()
{
	switch( header.tlc_version ) {
	case 0: return (new tlc_v0_frame(&header, &frames_queue_mutex, &frames_queue_updated)); break;
	case 1: return (new tlc_v1_frame(&header, &frames_queue_mutex, &frames_queue_updated)); break;
	default: return 0; break;
	}
}

bool tlc_writer::write_samples(int channel, short *data, unsigned int length)  //data must be of size samples_per_frame.
{
TCD
	//if we're finalizing the file, nobody better be trying to write samples!
	if(finish_write_thread) return false;

	wait_on_frame_writer(3);
				
	//lock block
	{
TCD		unique_lock<mutex> frames_queue_lock(frames_queue_mutex);

TCD		if( frames_queue[channel].empty() ) {
			frames_queue[channel].push_back( new_empty_frame() );
		}

TCD		while( length>0 ) {
TCD			tlc_frame *frame = frames_queue[channel].back();
			size_t samples_in_frame = frame->samples.size();
			size_t samples_to_copy = min(length, header.samples_per_frame - samples_in_frame);
			frame->samples.resize(samples_in_frame+samples_to_copy);
			memcpy( &frame->samples[samples_in_frame], data, samples_to_copy*sizeof(short) );
			length -= (unsigned int)samples_to_copy;
			data += samples_to_copy;
TCD			if(samples_in_frame+samples_to_copy >= header.samples_per_frame)
			{
#ifdef TLC_USE_FRAME_WRITE_THREAD_POOL
TCD				frame_thread_pool.schedule( bind(&tlc_frame::encode, frame) );
#else
TCD				frame->encode();
#endif

#ifndef TLC_USE_FILE_WRITE_THREAD
TCD				frames_queue_lock.unlock();
TCD				write_ready_frames();
TCD				frames_queue_lock.lock();
#endif

				//prepare for next frame
TCD				frames_queue[channel].push_back( new_empty_frame() );
TCD			}
TCD		}
TCD	}

TCD	update_frames(false);

TCD	return true;
}

bool tlc_writer::wait_on_frame_writer(unsigned int frame_count)
{
#ifndef TLC_USE_FILE_WRITE_THREAD
	write_ready_frames();
	return true;
#endif
TCD
	unique_lock<mutex> frames_queue_lock(frames_queue_mutex);
	
	unsigned int min_frame_count;
	do {
		min_frame_count = frame_count + 1;
		for(int c=0; c<header.channel_count; ++c) {
			min_frame_count = min( min_frame_count, frames_queue[c].size() );
		}
		if(min_frame_count > frame_count) {
			frames_queue_updated.wait( frames_queue_lock );
		}
	} while( min_frame_count > frame_count );
TCD
	return true;
}


unsigned int tlc_writer::get_sample_ready_count()
{	
	if( frames_queue[0].empty() ) {
		return 0;
	}

	//figure out how many samples are ready for writting
	// accross all channels
	unsigned int sample_count = frames_queue[0][0]->samples.size();

	for(int c=0; c<header.channel_count; ++c) {
		if( frames_queue[c].empty() ) {
			//this channel has no frames at all
			return 0;
		} else if( !frames_queue[c][0]->isFinished() ) {
			//this channel hasn't finished encoding it's front most frame
			return 0;
		} else if( sample_count != frames_queue[c][0]->samples.size() ) {
			//if frames accross channels have been encoded with
			// a different number of samples, they can not be written to the file
			return 0;
		}
	}

	return sample_count;
}


void tlc_writer::write_ready_frames()
{
	unsigned int sample_count = 0;

	do
	{
		{
			unique_lock<mutex> frames_queue_lock(frames_queue_mutex);
			sample_count = get_sample_ready_count();
		}


		//write a complete frame, if there are samples to write.
		if( sample_count>0 )
		{
			//keep track of the start of each frame
			unsigned long long channel_index_ptr = ofs.tellp();
			frame_index.push_back( channel_index_ptr );

			//initialize a new frame with a blank channel index
			unsigned int zero=0;
			for(int c=0; c<header.channel_count; c++) {
				ofs.write( (char*)&zero, sizeof(zero) );
			}

			//keep track of total samples per channel
			header.samples_per_channel += sample_count;

			//write the frames of each channel in order
			for(int c=0; c<header.channel_count; ++c)
			{
				//write the data to disk and discard it from RAM
TCD				tlc_frame *frame = frames_queue[c].front();
TCD				frame->write(ofs);
				{
					unique_lock<mutex> frames_queue_lock(frames_queue_mutex);
					frames_queue[c].pop_front();
				}
TCD				delete frame;
TCD				frames_queue_updated.notify_all();

				//update the channel index after writing the frame
				//end of this frame is the start of the next
				unsigned long long channel_ptr = ofs.tellp();
				unsigned int channel_offset =  (unsigned int) (channel_ptr - channel_index_ptr);
				ofs.seekp( (streamoff) channel_index_ptr + c*sizeof(channel_offset) );
				ofs.write( (char*)&channel_offset, sizeof(channel_offset) );
				ofs.seekp( channel_ptr );
TCD			}

			ofs.flush();
TCD		}
TCD	} while( sample_count != 0 );
}


//The worker thread function that takes incomming data
// and writes to the file.
void tlc_writer::operator()()
{
TCD	unsigned int sample_count = 0;
TCD	do
	{
TCD		//wait until we have a full frame ready or need to finalize
		do
		{
TCD			unique_lock<mutex> frames_queue_lock(frames_queue_mutex);
TCD			sample_count = get_sample_ready_count();
TCD			if(sample_count != header.samples_per_frame && !finish_write_thread) {
TCD				frames_queue_updated.wait( frames_queue_lock );
TCD			}
TCD		} while(sample_count != header.samples_per_frame && !finish_write_thread);

TCD		write_ready_frames();

	//keep on writing frames until we're done
TCD	} while( sample_count==header.samples_per_frame || !finish_write_thread );
TCD
	//warn about lost samples.
	/*
	for(int c=0; c<header.channel_count; ++c) {
		if( frames_queue[c].size() != 0 ) {
			unsigned int sample_count = frames_queue[c][0]->samples.size();
			if(sample_count > 0) {
				cout << "\nWarning: Channel " << c << " has "
					 << sample_count << " samples left.";
				cout.flush();
			}
		}
	}
	//*/
}

bool tlc_writer::update_frames(bool finalize)
{
TCD
	if(finalize)
	{
		//encode all un-encoded frames
TCD		for(int c=0; c<header.channel_count; ++c) {
			unique_lock<mutex> frames_queue_lock(frames_queue_mutex);
			if( !frames_queue[c].empty() ) {
				tlc_frame *frame = frames_queue[c].back();
				//check to see that the frame hasn't been encoded.
				if( frame->samples.size() < header.samples_per_frame ) {
#ifdef TLC_USE_FRAME_WRITE_THREAD_POOL
TCD					frame_thread_pool.schedule( bind(&tlc_frame::encode, frame) );
#else
TCD					frame->encode();
#endif
				}
			}
		}

		//make sure all frames have been encoded before finalizing
#ifdef TLC_USE_FRAME_WRITE_THREAD_POOL
TCD		frame_thread_pool.wait();
#endif

#ifndef TLC_USE_FILE_WRITE_THREAD
TCD		write_ready_frames();
#endif

		{
			unique_lock<mutex> frames_queue_lock(frames_queue_mutex);
			finish_write_thread = true;
TCD		}
TCD	}

TCD	frames_queue_updated.notify_all();
TCD	return true;
}


bool tlc_writer::write_frame_index()
{
TCD	//keep track of the offset for later
	header.frame_index_offset = ofs.tellp();

TCD	
	//write the frame count
	unsigned int frame_count = (unsigned int) frame_index.size();
	ofs.write( (char*)&frame_count, sizeof(frame_count) );
TCD	
	//write the frame offsets
	ofs.write( (char*)&frame_index[0],
	           (streamsize) (sizeof(frame_index[0])*frame_index.size()) );

TCD	return true;
}

//returns false if file doesn't end on a channel boundary.
bool tlc_writer::finalize()
{
TCD	//return if the file has already been finalized.
	if( !ofs.is_open() ) {
TCD		return true;
	}

	//write last frames (may be incomplete)
TCD	update_frames(true);
TCD	file_write_thread.join();

	//write the end of the file
TCD	write_frame_index();
TCD	ofs.write("TLCave",6);

	//update the header with final information
TCD	ofs.seekp( (streamoff) samples_per_channel_file_ptr );
TCD	ofs.write( (char*)&header.samples_per_channel, sizeof(header.samples_per_channel) );
TCD	ofs.seekp( (streamoff) frame_index_offset_file_ptr );
TCD	ofs.write( (char*)&header.frame_index_offset, sizeof(header.frame_index_offset) );

	//cout << "\nSamples written: " << header.samples_per_channel;

TCD	ofs.close();
TCD	return true;
}


/******************************************************************************
* tlc_reader
*/
tlc_reader::tlc_reader(const char *filename)
{
	//unique_lock<mutex> ifs_lock(ifs_mutex);
	ifs.open(filename, ios::binary | ios::in);	
	if( !header.read(ifs) ) return;

	//read the frame index or re-create it
	if( header.frame_index_offset != 0 )
	{
		ifs.seekg( (streamoff) header.frame_index_offset );
		unsigned int frame_count;
		ifs.read( (char*)&frame_count, sizeof(frame_count) );
		frame_index.resize( frame_count );
		ifs.read( (char*)&frame_index[0], frame_count * sizeof(frame_index[0]) );
	}
	else
	{
		cout << "TLC file " << filename << " not finalized properly.  Rebuilding index.\n";
		unsigned long long frame_offset = ifs.tellg();
		unsigned int next_frame_offset = 0;

		do
		{
			frame_index.push_back( frame_offset );
			ifs.seekg( frame_offset + (header.channel_count-1) * sizeof(next_frame_offset) );
			if(ifs.eof())
			{
					ifs.clear();
					ifs.seekg(0);
					break;
			}
			ifs.read( (char*)&next_frame_offset, sizeof(next_frame_offset) );
			frame_offset += next_frame_offset;
		} while( next_frame_offset!=0 );

		frame_index.pop_back();
		frame_index.pop_back(); //BCA TODO: why do i need to scratch two frames?  Only the last one should be bad.

		header.samples_per_channel = frame_index.size() * header.samples_per_frame;
	}

	//initialize frame cache
	frame_cache.resize( header.channel_count, 0 );
	frame_cache_index.resize( header.channel_count, 0 );
}

tlc_reader::~tlc_reader()
{
	//unique_lock<mutex> ifs_lock(ifs_mutex);
	ifs.close();
	for(size_t i=0; i<frame_cache.size(); ++i) {
		delete frame_cache[i];
	}
	
}

bool tlc_reader::read(int channel, unsigned long long first_sample_index, unsigned int length, short *data)
{
	unsigned long long last_sample_index = first_sample_index + length;
	unsigned int frame = (unsigned int) first_sample_index / header.samples_per_frame;
	unsigned int frame_offset = (unsigned int) first_sample_index % header.samples_per_frame;

	unsigned long long i=first_sample_index;
	while(i<last_sample_index)
	{
		read_frame(frame,channel);
		for( unsigned int j=frame_offset;
			 j<header.samples_per_frame && i<last_sample_index;
			 j++, i++ )
		{
			*data = frame_cache[channel]->getSample(j);
			data++;
		}
		frame++;
		frame_offset = 0;
	}
	return true;
}


bool tlc_reader::read_frame(unsigned int frame, int channel)
{
	if( frame_cache_index[channel] != frame ||
		frame_cache[channel] == 0 )
	{
		//make sure only one thread accesses the file at a time
		//unique_lock<mutex> ifs_lock(ifs_mutex);
		if( frame >= frame_index.size() ) {
			return false;
		}

		unsigned int channel_offset_begin = header.channel_count * sizeof(unsigned int);
		unsigned int channel_offset_end = 0;
		if(channel>=1) {
			ifs.seekg( (streamoff) (frame_index[frame] + (channel-1)*sizeof(unsigned int)) );
			ifs.read( (char*)&channel_offset_begin, sizeof(channel_offset_begin) );
		} else {
			ifs.seekg( (streamoff) (frame_index[frame] + (channel)*sizeof(unsigned int)) );
		}
		ifs.read( (char*)&channel_offset_end, sizeof(channel_offset_end) );
		ifs.seekg( streamoff(frame_index[frame] + channel_offset_begin) );

		unsigned int encoded_size = channel_offset_end - channel_offset_begin;
		vector<unsigned char> encoded_frame(encoded_size,0);
		ifs.read( (char*)&encoded_frame[0], encoded_size );
		if( !frame_cache[channel] ) {
			switch( header.tlc_version ) {
			case 0: frame_cache[channel] = new tlc_v0_frame(&header, &frames_queue_mutex, &frames_queue_updated); break;
			case 1: frame_cache[channel] = new tlc_v1_frame(&header, &frames_queue_mutex, &frames_queue_updated); break;
			default: return false; break;
			}
		}

		unsigned int sample_count = header.samples_per_frame;
		unsigned int max_frame = (unsigned int) header.samples_per_channel / header.samples_per_frame;
		if(frame==max_frame) {
			sample_count = (unsigned int) header.samples_per_channel % header.samples_per_frame;
		}
		frame_cache[channel]->decode(encoded_frame, sample_count);
		frame_cache_index[channel] = frame;
	}
	return true;
}

