/******************************************************************************
Tecella Lossless Compression Format
Copyright (c) 2010 Tecella LLC
Licensed under the MIT License

*******************************************************************************
File Description:
Provides the framework for different versions of the TLC file format.

*******************************************************************************
Contributors:
Brian Anderson (Tecella) - Initial implementation

******************************************************************************
TODO:
1) Code assumes platform uses little endian byte order.  Don't do that.
******************************************************************************/


#ifndef __TECELLA_TLC_IMPLEMENTATION__
#define __TECELLA_TLC_IMPLEMENTATION__

#include <iostream>

#include <vector>
#include <deque>
using namespace std;


//TLC_USE_FILE_WRITE_THREAD: indicates if a separate thread should be used
// to write data to disk so the calling thread doesn't have to wait block
#define TLC_USE_FILE_WRITE_THREAD

//TLC_USE_FRAME_WRITE_THREAD_POOL: indicates if a thread pool should be used
// to encode the incomming samples, so the calling thread doesn't have to block.
// Not very useful without TLC_USE_FILE_WRITE_THREAD also defined.
#define TLC_USE_FRAME_WRITE_THREAD_POOL

//TLC_USE_FRAME_BLOCK_READ_THREADS: indicates fine-grain parallelism should
// be used to decode a single frame.  Current implementations use
// a block of threads.  Future implementation may use CUDA or OpenCL.
#define TLC_USE_FRAME_BLOCK_READ_THREADS

//TLC_USE_FRAME_BLOCK_WRITE_THREADS: indicates fine-grain parallelism should
// be used to encode a single frame.
//Note: TLC_USE_FRAME_WRITE_THREAD_POOL usually provides sufficient
// parallelism.  This is mainly for testing in anticipation of using CUDA or OpenCL.
//#define TLC_USE_FRAME_BLOCK_WRITE_THREADS


/*
#include <fstream>
//*/
//*
#include "tc_fstream.hpp"
#define ofstream tc_ofstream
#define ifstream tc_ifstream
//*/

#ifdef TLC_USE_FRAME_WRITE_THREAD_POOL
//#include <boost/threadpool.hpp>  //not officially part of the boost library
#include "tlc_thread_pool.h"
#endif

//*
//If we are using boost type threads...
#include <boost/thread/thread.hpp>
#include <boost/thread/mutex.hpp>
#include <boost/thread/condition.hpp>
#define condition_variable condition
using namespace boost;
//*/
/*
//If we are using C++0x type threads...
#include <thread>
#include <mutex>
#include <condition_variable>
typedef condition_variable condition;
//*/


//#define DEBUG
extern mutex cout_mutex;
#ifdef DEBUG
#	define TCD { unique_lock<mutex> cout_lock(cout_mutex); \
	             std::cout << __FILE__ << " " << __PRETTY_FUNCTION__ << " " \
	                     << __LINE__ << std::endl; std::cout.flush(); }
#else
#	define TCD
#endif

#define TLC_DEBUG_FUNCLINE
//#define TLC_DEBUG_FUNCLINE cout << __FUNCTION__ << " " << __LINE__ << endl; cout.flush();

//Forward declarations
class tlc_frame;
class tlc_file_header;

//Useful functions

//returns the minimum number of bits needed to represent i in two's complement
unsigned char bits_needed(int i);     // reports 0 as needing 1 bit.
unsigned char bits_needed_v1(int i);  // reports 0 as needing 0 bits.


/******************************************************************************
* tlc_frame
* The virtual class to be implemented by different types of frames
*/
class tlc_frame
{
friend class tlc_writer;
protected:
	//helper variables
	tlc_file_header *file_header;

	vector<short> samples;

	//asynchronous messaging
	//at then end of encode/decode:
	//  the mutex is locked, finished is set, and the condition is notified
	mutex *frames_mutex;
	condition_variable *frames_condition;
	bool finished;

public:
	tlc_frame(tlc_file_header *file_header, mutex *frames_mutex, condition_variable *frames_condition)
		: file_header(file_header), frames_mutex(frames_mutex),
		  frames_condition(frames_condition), finished(false)
	{}
	virtual ~tlc_frame() {}

	//returns false if read failed, true otherwise
	virtual bool decode(vector<unsigned char> &encoded_data, unsigned int sample_count) = 0;

	//after read has been called, you can access the data using getSample
	virtual short getSample(size_t i) = 0;

	//encode overwrites samples to perform an in-place encoding
	virtual void encode() = 0;

	//write writes the encoded data to the file
	virtual void write(ofstream &ofs) = 0;

	bool isFinished() { return finished; }
};


/******************************************************************************
* tlc_file_header
*/
struct tlc_file_header
{
public:
	//front of file
	unsigned char  tlc_version;
	unsigned short channel_count;
	unsigned char freq_or_period_mode;
	double  freq_or_period;
	unsigned char  bits_per_sample;
	unsigned long long  samples_per_channel;  //0 indicates file is not done writing.
	unsigned int   samples_per_frame;
	unsigned long long  frame_index_offset;
	vector<char>   metadata;
	vector<double> channel_scales;
	vector<string> channel_labels;

public:
	//returns true if success, false otherwise
	bool write(ofstream &ofs);
	bool read(ifstream &ifs);
};


/******************************************************************************
* tlc_writer
*/
class tlc_writer
{
private:
	bool init_error;

	ofstream ofs;

	tlc_file_header header;

	//frame_index gets emptied every frame interval
	vector< unsigned long long > frame_index;
	vector< deque< tlc_frame* > > frames_queue;  //[channel][frame]

	unsigned long long samples_per_channel_file_ptr;
	unsigned long long frame_index_offset_file_ptr;

	//thread stuff
#ifdef TLC_USE_FRAME_WRITE_THREAD_POOL
	//threadpool::fifo_pool frame_thread_pool;
	thread_pool frame_thread_pool;
#endif
	bool finish_write_thread;
	thread file_write_thread;
	mutex frames_queue_mutex;   //should be locked when accessing frames_queue or finish_write_thread
	condition_variable frames_queue_updated;

public:
	tlc_writer( unsigned int   version,
	            const char*    filename,
	            unsigned short channel_count,
				bool           interpret_as_period,
				double         sample_freq_or_period,
				unsigned char  bits_per_sample,
				unsigned int   samples_per_frame );

	~tlc_writer();

	bool set_metadata(char *data, unsigned int length);
	bool set_channel_scale(int channel, double scale);
	bool set_channel_label(int channel, const char *label, int length);
	bool start_writing_frames();
	tlc_frame* new_empty_frame();
	bool write_samples(int channel, short *data, unsigned int length);
	bool wait_on_frame_writer(unsigned int frame_count);
	bool finalize();

public:
	//returns how many samples can be "tetris'd" out
	unsigned int get_sample_ready_count();
	//thread function
	void operator()();
	void write_ready_frames();

protected:
	bool update_frames(bool finalize);
	bool write_frame_index();
};

/******************************************************************************
* tlc_reader
*/
class tlc_reader
{
private:
	ifstream ifs;

	tlc_file_header header;
	vector< unsigned long long > frame_index;
	vector< tlc_frame* > frame_cache;			//one per channel for now
	vector< unsigned int > frame_cache_index;

	//thread stuff
	thread file_read_thread;
	mutex frames_queue_mutex;
	condition_variable frames_queue_updated;

public:
	tlc_reader(const char *filename);
	~tlc_reader();

	const tlc_file_header& getHeader() { return header;  }
	bool read(int channel, unsigned long long first_sample_index, unsigned int length, short *data);
	bool read_frame(unsigned int frame, int channel);
};

#endif
