#include <iostream>
#include <fstream>

#include <vector>
using namespace std;

#include <boost/program_options/cmdline.hpp>
#include <boost/program_options/environment_iterator.hpp>
#include <boost/program_options/eof_iterator.hpp>
#include <boost/program_options/errors.hpp>
#include <boost/program_options/option.hpp>
#include <boost/program_options/options_description.hpp>
#include <boost/program_options/parsers.hpp>
#include <boost/program_options/positional_options.hpp>
#include <boost/program_options/value_semantic.hpp>
#include <boost/program_options/variables_map.hpp>
#include <boost/program_options/version.hpp>
namespace po = boost::program_options;

#include <boost/progress.hpp>
#include <boost/algorithm/string.hpp>
using namespace boost;

#include "TecellaTLC.h"

void tlc2atf(string &infile, string &outfile, vector<int> &channels,
			 unsigned long long start_index, unsigned long long end_index,
			 unsigned int eat_samples, unsigned long long split_samples);

int main(int argc, char *argv[])
{
	cout << "Tecella Lossless Compression (TLC) format to ATF converter.\n";
	cout << "Copyright 2008 Tecella LLC.\n";

	vector<int> channels;
	string infile;
	string outfile;

	unsigned long long start_index = 0;
	unsigned long long end_index = 0xFFFFFFFFFFFFFFFF;
	unsigned int eat_samples = 1;
	unsigned long long split_samples = 0;

	// Declare the supported options.
	po::options_description desc("\nTo convert an entire TLC file, you can drag and drop it onto the tlc2atf.exe.  For more control, you can use the command line.\n\
\nExamples:\n\
\  tlc2atf capture.tlc\n\t : Converts all samples from all channels to capture.atf\n\
\  tlc2atf -i capture.tlc -o test.atf -c 1 -c 2\n\t : Converts all samples from channels 1 and 2 to test.atf.\n\
\  tlc2atf -i capture.tlc -s 0 -e 999\n\t : Converts the first 1000 samples of all channels into capture.atf.\n\
\  tlc2atf -i capture.tlc -s 10000 -x 4\n\t : Converts every 4th sample from all channels to a series of files captureXXX.atf.  Each file in the series will have no more than 10000 samples per channel.\n\
\n\nAllowed options:\n\
\nNote: You man use either the full --<option> format or the shorthand -<o> format.");
	desc.add_options()
		("help,h", "Display this help message.")
		("channel,c", po::value< vector<int> >(&channels), "Specify a channel to extract.  Absense of this option implies all channels will be extracted.  To extract multiple channels use \"-c # ... -c #\".")
		("outfile,o", po::value<string>(&outfile), "Output ATF file.  Defaults to same filename with atf extension.")
		("infile,i", po::value<string>(&infile), "Input TLC filename.")
		("start_index,s", po::value< unsigned long long >(&start_index), "First sample to extract, inclusive.  Samples indexed starting from 0.")
		("end_index,e", po::value< unsigned long long >(&end_index), "Last sample to extract, inclusive.  Samples indexed starting from 0.")
		("split,v", po::value< unsigned long long >(&split_samples), "How many samples per channel to put into an ATF before automatically spliting the file.")
		("eat_samples,x", po::value< unsigned int >(&eat_samples), "Reduces the original sample rate by this multiple. (must be >= 1)")
	;

	po::positional_options_description p;
	p.add("infile", -1);

	po::variables_map vm;
	po::store(po::command_line_parser(argc, argv).
			  options(desc).positional(p).run(), vm);
	po::notify(vm);

	// Make sure an input file is specified
	if( !vm.count("infile") || eat_samples<=0 ) {
		cout << desc << endl;
		return 1;
	}

	// Decide what output filename we should use
	if( !vm.count("outfile") ) {
		outfile = infile;
		if( iends_with(infile, ".tlc") ) {
			replace_tail(outfile, 3, "atf");
		} else {
			outfile += ".atf";
		}
	}

	cout << start_index << " " << end_index << " " << eat_samples << " " << split_samples  << "\n";

	// Convert!
	{
		//progress_timer t;
		tlc2atf(infile, outfile, channels, start_index, end_index, eat_samples, split_samples);
	}

	return 1;
}


void tlc2atf(string &infile, string &outfile, vector<int> &channels,
			 unsigned long long start_index, unsigned long long end_index,
			 unsigned int eat_samples, unsigned long long split_samples)
{
	TLC_READ_HANDLE tlcrh;
	TLC_INFO tlc_info;
	tlcr_open( &tlcrh, infile.c_str() );
	tlcr_get_info( tlcrh, &tlc_info );

	if(end_index == 0xFFFFFFFFFFFFFFFF) {
		end_index = tlc_info.samples_per_channel;
	}

	if(eat_samples<=0) {
		eat_samples = 1;
	}

	if( channels.empty() ) {
		for(int i=1; i<=tlc_info.channel_count; ++i) {
			channels.push_back(i);
		}
	}

	int atf_split_number = 0;

	while( start_index < end_index )
	{
		unsigned long long end_index2 = end_index;
		string outfile2 = outfile;

		if(split_samples > 0)
		{
			if( ((end_index - start_index) / eat_samples) > split_samples ) {
				end_index2 = start_index + eat_samples*split_samples;
			}
			char temp[128];
			sprintf(temp,"%03d.atf",atf_split_number);
			replace_tail(outfile2, 3, temp);
		}

		ofstream ofs(outfile2.c_str());

		string sep = "\t";

		ofs << "ATF" << sep << "1.0" << endl;
		ofs << "4" << sep << channels.size() << endl;

		//Optional Headers
		ofs << "\"TecellaDataVersion=0.2\"\n";
		ofs << "\"TecellaChannelCount=" << channels.size() << "\"\n";
		if(tlc_info.interpret_as_period) {
			ofs << "\"TecellaSamplePeriod=" << tlc_info.period * (eat_samples) << "\"\n";
		} else {
			ofs << "\"TecellaSampleFrequency=" << tlc_info.frequency / (eat_samples) << "\"\n";
		}
		ofs << "\"TecellaSampleUnits=pA\"\n";

		//Column headers
		for(size_t c=0; c<channels.size(); c++) {
			const int MAX_LABEL_LENGTH = 64;
			char label[MAX_LABEL_LENGTH];
			tlcr_get_channel_label(tlcrh, channels[c]-1, label, MAX_LABEL_LENGTH);
			ofs << "\"" << label << "\"";
			if( c+1 != channels.size() ) {
				ofs << sep;
			}
		}
		ofs << endl;
		
		ofs.precision(5);
		ofs.setf(ios::fixed,ios::floatfield);

		//Output the samples
		const int MAX_MSG_LENGTH = 512;
		char msg[MAX_MSG_LENGTH];
		for(unsigned long long i=start_index; i<end_index2; i+=eat_samples)
		{
			//Periodically update user with progress.
			if( (i&0x7FF) == 0 || i==tlc_info.samples_per_channel-1) {
				size_t msg_length = strlen(msg);
				sprintf(msg, "Converting sample: %lld / %lld", i+1, tlc_info.samples_per_channel);
				for(size_t j=0; j<msg_length; ++j) {
					cout << "\b";
				}
				cout << msg;
			}

			//Print the current sample for each channel.
			for(size_t c=0; c<channels.size(); c++) {
				double value;
				tlcr_read_d(tlcrh, channels[c]-1, i, 1, &value);
				ofs << value * 1e12;
				if( c+1 != channels.size() ) {
					ofs << sep;
				}
			}
			ofs << endl;
		}

		start_index = end_index2;
		atf_split_number++;
		ofs.close();
	}

	tlcr_close(tlcrh);
}
