/******************************************************************************
Tecella Lossless Compression Format
Copyright (c) 2010 Tecella LLC
Licensed under the MIT License

*******************************************************************************
File Description:

Implements a simple thread pool that queues up tasks and
   executes them using a constant number of recycled threads.

The purpose of a thread pool is to:
	1) avoid the overhead of creating and destroying threads and
	2) prevent too many threads from running simultaneously.

The thread_pool class should be the only class of interest from an interface
   perspective.

*******************************************************************************
Contributors:
Brian Anderson (Tecella) - Initial implementation

******************************************************************************/

#ifndef __TLC_THREAD_POOL__
#define __TLC_THREAD_POOL__

/*******************************************
Includes
*******************************************/
#include <functional>
#include <deque>
using namespace std;

#include <boost/thread/thread.hpp>
#include <boost/thread/mutex.hpp>
#include <boost/thread/condition.hpp>
using namespace boost;


/*******************************************
Forward declarations
*******************************************/
class thread_pool_worker;


/*******************************************
Typedefs
*******************************************/
typedef std::function<void()> TaskType;


/******************************************************************************
Class: thread_pool_data
Inherited by thread_pool.
Provides a common interface for the thread_pool 
   and thread_pool_worker classes to communicate.
Also makes it so that we don't need a separate *.cpp implementation file.
******************************************************************************/

class thread_pool_data
{
friend class thread_pool_worker;
protected:

	//the pool of threads
	vector<thread*> threads;
	
	//a flag to indicate the thread pool should finish up
	volatile bool finish_worker_threads; 

	//a queue of tasks
	deque<TaskType> tasks;

	//synchronization primitives for access to:
	// finish_worker_threads and tasks.
	mutex tasks_mutex;
	condition tasks_updated;
};


/******************************************************************************
Class: thread_pool_worker
Represents a single thread within the pool.
It waits for a new task, claims it from the pool, then executes it.
******************************************************************************/

class thread_pool_worker
{
private:
	TaskType task;
	thread_pool_data *parent;

public:
	//"regular" constructor
	thread_pool_worker(thread_pool_data *parent)
	: task(), parent(parent)
	{}

	//copy constructor
	thread_pool_worker(const thread_pool_worker &src)
	: task(src.task), parent(src.parent)
	{}

	//the worker thread
	void operator() ()
	{
		while(true)
		{
			//get a queued task
			{
				unique_lock<mutex> tasks_lock( parent->tasks_mutex );

				//wait for a task if there aren't any
				while( parent->tasks.empty() && !parent->finish_worker_threads ) {
					parent->tasks_updated.wait( tasks_lock );
				}

				//if tasks is empty, we can only get here if
				// kill_worker_threads is true, so exit.
				if( parent->tasks.empty() ) return;

				//claim the task and remove it from the queue
				task = parent->tasks.front();
				parent->tasks.pop_front();
			}

			//execute the task
			task();
		}
	}
};


/******************************************************************************
Class: thread_pool
This is the main class used to manage a thread pool.
******************************************************************************/

class thread_pool : public thread_pool_data
{	
public:
	//constructs a thread pool with the specified number of worker threads
	thread_pool(int pool_size)
	{
		finish_worker_threads = false;

		thread_pool_worker worker( this );
		for(int i=0; i<pool_size; i++) {
			threads.push_back( new thread(worker) );
		}
	}

	//destructor
	~thread_pool()
	{
		wait();
		for(size_t i=0; i<threads.size(); ++i) {
			delete threads[i];
		}
	}

	//adds a task to the queue
	void schedule( TaskType task )
	{
		{
			unique_lock<mutex> tasks_lock( tasks_mutex );
			tasks.push_back(task);
		}
		tasks_updated.notify_one();
	}

	//waits for all scheduled tasks to complete
	// and closes the worker threads
	void wait()
	{
		//return if we already killed the threads
		if(finish_worker_threads) return;

		//notify worker threads to finish up
		{
			unique_lock<mutex> tasks_lock( tasks_mutex );
			finish_worker_threads = true;
		}
		tasks_updated.notify_all();

		//wait for worker threads to finish
		for(size_t i=0; i<threads.size(); ++i) {
			threads[i]->join();
		}

	}
};


#endif // __TLC_THREAD_POOL__
