/******************************************************************************
Tecella Lossless Compression Format
Copyright (c) 2010 Tecella LLC
Licensed under the MIT License

*******************************************************************************
File Description:
Implements TLCv0, which has the following attributes:
	3% to 5% better compression than TLCv1.
	More complicated than TLCv1.
	Slower compression than TLCv1.
	Parallelizable only at a macro scale:
		Many frames with 1 thread per frame.
	Whereas TLCv1 is parallelizable at a micro scale:
		Many frames with many threads per frame.

See documentation for details about the encoding scheme.

*******************************************************************************
Contributors:
Brian Anderson (Tecella) - Initial implementation

******************************************************************************
TODO:  N/A
******************************************************************************/

#ifndef __TECELLA_TLC_V0_IMPLEMENTATION__
#define __TECELLA_TLC_V0_IMPLEMENTATION__

#include <vector>
using namespace std;

#include "tlc.h"

class tlc_v0_frame;

/******************************************************************************
* tlc_bit_packer
*/
//packs values of aritrary number of bits tightly into bytes
class tlc_bit_packer
{
private:
	unsigned int bits_left;			//bits left in last byte
	vector<unsigned char> *buffer;

public:
	//takes an external queue the packed bytes will be pushed into
	tlc_bit_packer(vector<unsigned char> *buffer);

	//writes the least significant bits of data to the queue
	void write_bits(unsigned int data, unsigned int bits);
};


/******************************************************************************
* tlc_bit_unpacker
*/
//reads bits from tightly packed data
class tlc_bit_unpacker
{
private:
	unsigned long long data_out;
	unsigned int data_valid_bits;
	size_t i_data_in;
	vector<unsigned char> &data_in;

public:
	tlc_bit_unpacker(vector<unsigned char> &data);

	//returns true if bits were read
	//false if there weren't enough bits
	//pop indicates if the data read should be removed
	bool read_bits(unsigned int *data, unsigned int bits, bool pop=true);
	bool read_bits_se(int *data, unsigned int bits, bool pop=true);  //sign extended
};


/******************************************************************************
* tlc_v0_sub_frame
*/
//sub frame base class 

class tlc_v0_sub_frame
{
protected:
	tlc_v0_frame *parent_frame;

public:
	virtual unsigned char get_method_id() = 0;

	//decodes encoded_data into data.
	virtual void decode(tlc_bit_unpacker &bit_unpacker, vector<short> &data) = 0;

	//encodes data into encoded_data.
	virtual void encode(const short *data, int sample_count, tlc_bit_packer &bit_packer) = 0;
};


/******************************************************************************
* tlc_v0_sub_frame_t1
*/
//Sub Frame Type 1: Differential fixed width
//For spec v0 only
class tlc_v0_sub_frame_t1 : public tlc_v0_sub_frame
{
private:
	int bits_per_diff;

public:
	tlc_v0_sub_frame_t1(tlc_v0_frame *parent, int bits_per_diff=-1);

	virtual unsigned char get_method_id();
	virtual void decode(tlc_bit_unpacker &bit_unpacker, vector<short> &data);
	virtual void encode(const short *data, int sample_count, tlc_bit_packer &bit_packer);
};


/******************************************************************************
* tlc_v0_frame
*/

class tlc_v0_frame : public tlc_frame
{
friend class tlc_v0_sub_frame;
friend class tlc_v0_sub_frame_t1;

private:
	//unsigned int frame_size;
	vector<unsigned char> encoded_data;

	//for read/decode only
	//vector<short> samples;

	//variables that are actually written to file.
	unsigned char subframe_count;
	vector<tlc_v0_sub_frame*> subframes;

public:
	tlc_v0_frame(tlc_file_header *file_header, mutex *frames_mutex, condition_variable *frames_condition);
	virtual ~tlc_v0_frame();

	short getSample(size_t i) { return samples[i]; }

	void reset_subframes();

	//returns false if read failed, true otherwise
	bool decode(vector<unsigned char> &encoded_data, unsigned int sample_count);

	//encode overwrites samples to perform an in-place encoding
	void encode();

	//write writes the encoded data to the file
	void write(ofstream &ofs);
};


#endif
