/******************************************************************************
Tecella Lossless Compression Format
Copyright (c) 2010 Tecella LLC
Licensed under the MIT License

*******************************************************************************
File Description:
Implements TLCv1, which has the following attributes:
	Faster compression than TLCv0.
	3% to 5% worse compression than TLCv1.
	Simpler than TLCv0.
	Parallelizable at a micro scale:
		Many frames with many threads per frame.
		Easier to accelerate for SIMD architectures.
	Whereas TLCv0 is only parallelizable at a macro scale:
		Many frames with 1 thread per frame.
		Difficult to accelerate for SIMD architectures.

TLCv1 was designed with GPU acceleration in mind.

See documentation for details about the encoding scheme.

*******************************************************************************
Contributors:
Brian Anderson (Tecella) - Initial implementation

*******************************************************************************
TODO:
1) Add CUDA support.
2) Add OpenCL support.

******************************************************************************/

#ifndef __TECELLA_TLC_V1_IMPLEMENTATION__
#define __TECELLA_TLC_V1_IMPLEMENTATION__

#include <vector>
using namespace std;

#include "tlc.h"


/******************************************************************************
* tlc_v1_sub_frame
*/
class tlc_v1_sub_frame
{
private:
	//helper variables
	tlc_file_header *file_header;

public:
	//for encode only
	vector<unsigned char> encoded_header;
	unsigned char *encoded_diffs;
	unsigned int encoded_diffs_size;

public:
	tlc_v1_sub_frame(tlc_file_header *file_header);
	virtual ~tlc_v1_sub_frame();

	//returns false if read failed, true otherwise
	bool decode( unsigned char *encoded_data, unsigned int size,
				 short *samples, unsigned int sample_count );

	//encode overwrites samples to perform an in-place encoding of deltas
	//returns the size of the encoded frame
	//header will exist in sub_frame_header
	unsigned int encode(short *samples, unsigned int sample_count);

	bool write(ofstream &ofs);
};


/******************************************************************************
* tlc_v1_frame
*/
class tlc_v1_frame : public tlc_frame
{
private:

	//for write/encode only
	vector<char> subframe_sizes;
	vector<char> encoded_headers;
	vector<unsigned int> encoded_headers_sizes;
	vector<unsigned int> encoded_diffs_sizes;


public:
	tlc_v1_frame(tlc_file_header *file_header, mutex *frames_mutex, condition *frames_condition);
	virtual ~tlc_v1_frame();

	short getSample(size_t i) { return samples[i]; }

	//returns false if read failed, true otherwise
	bool decode(vector<unsigned char> &encoded_data, unsigned int sample_count);

	//encode overwrites samples to perform an in-place encoding
	void encode();

	//write writes the encoded data to the file
	void write(ofstream &ofs);
};

#endif
